### MACHINE LEARNING ESSENTIALS: PRACTICAL GUIDE IN R
### https://www.datanovia.com/en/product/machine-learning-essentials-practical-guide-in-r/

# A) Unsupervised learning methods,
#    to explore and discover knowledge
#    from a large multivariate data set
#    using clustering and principal component methods.
#     - hierarchical clustering
#     - k-means
#     - principal component analysis
#     - correspondence analysis

# B) Regression analysis,
#    to predict a quantitative outcome value
#    using linear regression and non-linear regression
#    strategies.

# C) Classification techniques,
#    to predict a qualitative outcome value
#    using logistic regression,
#          discriminant analysis,
#          naive bayes classifier,
#          and support vector machines.

# D) Advanced machine learning methods,
#    to build robust regression and classification models
#    using k-nearest neighbors methods,
#          decision tree models,
#          ensemble methods (bagging, random forest and boosting).

# E) Model selection methods,
#    to select automatically the best combination
#    of predictor variables
#    for building an optimal predictive model.
#    These include,
#          best subsets selection methods,
#          stepwise regression
#          and penalized regression (ridge, lasso and elastic net regression models).
#          principal component-based regression methods, which are useful when the data contain multiple correlated predictor variables.

# F) Model validation and evaluation techniques
#    for measuring the performance of a predictive model.

# G) Model diagnostics
#    for detecting and fixing a potential problems
#    in a predictive model.


###########################################################
##                Discriminant analysis                  ##
###########################################################

### http://www.sthda.com/english/articles/36-classification-methods-essentials/146-discriminant-analysis-essentials-in-r/

# Discriminant analysis is used
# to predict the probability
# of belonging to a given class (or category)
# based on one or multiple predictor variables.

# It works with continuous and/or categorical
# predictor variables.

# logistic regression for two-class classification problems,
# that is when the outcome variable has two possible values
# (0/1, no/yes, negative/positive).

# The discriminant analysis is more suitable
# for predicting the category of an observation
# in the situation where the outcome variable contains
# more than two classes. 

# Note that, both logistic regression and discriminant analysis
# can be used for binary classification tasks.

## discriminant analysis methods:

# Linear discriminant analysis (LDA):
#    Uses linear combinations of predictors
#    to predict the class of a given observation.
#    Assumes that
#    (1) the predictor variables (p) are normally distributed
#    (2) the classes have identical variances (for univariate
#        analysis, p=1) or identical covariance matrices (for
#        multivariate analysis, p > 1).

# Quadratic discriminant analysis (QDA):
#    More flexible than LDA.
#    Here, there is no assumption that the covariance matrix
#    of classes is the same.

# Mixture discriminant analysis (MDA):
#    Each class is assumed to be a Gaussian mixture
#    of subclasses.

# Flexible Discriminant Analysis (FDA):
#    Non-linear combinations of predictors is used
#    such as splines.

# Regularized discriminant anlysis (RDA):
#    Regularization (or shrinkage) improves the estimate
#    of the covariance matrices in situations
#    where the number of predictors is larger than
#    the number of samples in the training data.
#    This leads to an improvement of the discriminant
#    analysis.

###(1) Loading required R packages
install.packages("tidyverse")  # for easy data manipulation and visualization
install.packages("caret")  # for easy machine learning workflow
library(tidyverse)
library(caret)
theme_set(theme_classic())

###(2) Preparing the data

# iris data set:
# for predicting iris species
# based on the predictor variables
# Sepal.Length, Sepal.Width, Petal.Length, Petal.Width.

# Discriminant analysis can be affected by the scale/unit
# in which predictor variables are measured. Its generally
# recommended to standardize/normalize continuous predictor
# before the analysis.

##(2)-1 Split the data into training and test set:

# Load the data
data("iris")

# Split the data into training (80%) and test set (20%)
set.seed(123)
training.samples <- iris$Species %>%
  createDataPartition(p=0.8, list=FALSE)
train.data <- iris[ training.samples, ]
test.data  <- iris[-training.samples, ]

##(2)-2 Normalize the data. Categorical variables are automatically ignored.

# Estimate preprocessing parameters
preproc.param <- train.data %>% 
  preProcess(method=c("center","scale"))

# Transform the data using the estimated parameters
train.transformed <- preproc.param %>% predict(train.data)
test.transformed  <- preproc.param %>% predict(test.data)

###(3) Linear discriminant analysis - LDA

# The LDA algorithm starts by finding directions
# that maximize the separation between classes,
# then use these directions to predict the class
# of individuals. These directions, called linear
# discriminants, are a linear combinations of
# predictor variables.

# LDA assumes that
#  (1) predictors are normally distributed 
#  (2) the different classes have class-specific
#      means and equal variance/covariance.

# Before performing LDA, consider:
# Inspecting the univariate distributions of each variable
# and make sure that they are normally distribute.
# If not, you can transform them 
# using log and root for exponential distributions
# and Box-Cox for skewed distributions.
# removing outliers from your data and
# standardize the variables to make their scale comparable.

# The linear discriminant analysis can be easily computed
# using the function lda() [MASS package].

library(MASS)

# Fit the model
model <- lda(Species~., data=train.transformed)
model
plot(model)

# Make predictions
predictions <- model %>% predict(test.transformed)
names(predictions)

# Predicted classes
head(predictions$class, 6)

# Predicted probabilities of class memebership.
head(predictions$posterior, 6) 

# Linear discriminants
head(predictions$x, 3) 

lda.data <- cbind(train.transformed, predict(model)$x)
ggplot(lda.data,
       aes(LD1, LD2)) +
       geom_point(aes(color=Species))

# Model accuracy
mean(predictions$class==test.transformed$Species)

# Note that, if the predictor variables are standardized
# before computing LDA, the discriminator weights can be
# used as measures of variable importance for feature
# selection.

###(4) Quadratic discriminant analysis - QDA

# QDA is little bit more flexible than LDA, in the sense
# that it does not assume the equality of variance/covariance.
# In other words, for QDA the covariance matrix can be
# different for each class.

# LDA tends to be a better than QDA when you have a small
# training set.

# In contrast, QDA is recommended if the training set is
# very large, so that the variance of the classifier is not
# a major issue, or if the assumption of a common covariance
# matrix for the K classes is clearly untenable
# (James et al. 2014).

# QDA can be computed using the R function qda()
# [MASS package]

library(MASS)

# Fit the model
model <- qda(Species~., data=train.transformed)
model

# Make predictions
predictions <- model %>% predict(test.transformed)

# Model accuracy
mean(predictions$class == test.transformed$Species)

###(5) Mixture discriminant analysis - MDA

# The LDA classifier assumes that each class comes from
# a single normal (or Gaussian) distribution. This is
# too restrictive.

# For MDA, there are classes, and each class is assumed
# to be a Gaussian mixture of subclasses, where each data
# point has a probability of belonging to each class.
# Equality of covariance matrix, among classes, is still
# assumed.

install.packages("mda")
library(mda)

# Fit the model
model <- mda(Species~., data=train.transformed)
model

# Make predictions
predicted.classes <- model %>% predict(test.transformed)

# Model accuracy
mean(predicted.classes == test.transformed$Species)

###(6) Flexible discriminant analysis - FDA

# FDA is a flexible extension of LDA that uses non-linear
# combinations of predictors such as splines. FDA is useful
# to model multivariate non-normality or non-linear
# relationships among variables within each group,
# allowing for a more accurate classification.

library(mda)

# Fit the model
model <- fda(Species~., data=train.transformed)
model

# Make predictions
predicted.classes <- model %>% predict(test.transformed)

# Model accuracy
mean(predicted.classes == test.transformed$Species)

###(7) Regularized discriminant analysis

# RDA builds a classification rule
# by regularizing the group covariance matrices
# (Friedman 1989) allowing a more robust model
# against multicollinearity in the data.
# This might be very useful
# for a large multivariate data set
# containing highly correlated predictors.

# Regularized discriminant analysis is a kind of
# a trade-off between LDA and QDA. Recall that,
# in LDA we assume equality of covariance matrix
# for all of the classes. QDA assumes different
# covariance matrices for all the classes.
# Regularized discriminant analysis is
# an intermediate between LDA and QDA.

# RDA shrinks the separate covariances of QDA
# toward a common covariance as in LDA.
# This improves the estimate of the covariance
# matrices in situations where the number of
# predictors is larger than the number of samples
# in the training data, potentially leading to
# an improvement of the model accuracy.

install.packages("klaR")
library(klaR)

# Fit the model
model <- rda(Species~., data = train.transformed)
model

# Make predictions
predictions <- model %>% predict(test.transformed)

# Model accuracy
mean(predictions$class == test.transformed$Species)



####################################################
### Discriminant Analysis - bankloan       ###
####################################################
data4 <- read.csv("C:/Users/leep/Desktop/û2019/ٺ2019/bankloan1.csv")
data4 <- data4[,-10]
colnames(data4) <- c("","","","ֳ",
                     "ҵ","ä","īкä","Ÿä",
                     "ü̷")
data4$ <- factor(data4$,
                     levels=c(1:5),
                     labels=c("̸","","",
                              "","п"))
data4$ü̷ <- factor(data4$ü̷,
                     levels=c(0,1),
                     labels=c("(=췮)","(=ҷ)"))
str(data4)

X <- data4[1:700,]; Y <- data4[701:850,]
mean_out <- aggregate(X[,-c(2,9)],by=list(ü̷=X$ü̷),FUN=mean)
print(mean_out, digits=3)
(зǥ_ <- xtabs(~+ü̷,data=X))
addmargins(зǥ_,margin=2)
(зǥ_ <- prop.table(зǥ_,margin=1))
addmargins(зǥ_,margin=2)
chisq.test(x=X$,y=X$ü̷)

t.test(~ü̷,X)
t.test(~ü̷,X)
t.test(ֳ~ü̷,X)
t.test(ҵ~ü̷,X)
t.test(ä~ü̷,X)
t.test(īкä~ü̷,X)
t.test(Ÿä~ü̷,X)

newX <- X[,-2]
#install.packages("MVTests")
#library(MVTests)
BoxM(newX[,-8],newX$ü̷)

#install.packages("biotools")
#library(biotools)
boxM(newX[,-8],newX$ü̷)  

## Ϻ  ׳ lda() غ...
install.packages("MASS")
library(MASS)
lda_out <- lda(ü̷~., data=newX)
str(lda_out)
addmargins(xtabs(~newX$ü̷+predict(lda_out)$class), margin=2)
addmargins(prop.table(xtabs(~newX$ü̷+predict(lda_out)$class),margin=1),
           margin=2)

##  Ʒÿ  7:3   lda() غ...
install.packages("caTools")
library(caTools)
set.seed(12345)
split <- sample.split(newX$ü̷, SplitRatio = 0.7)
Ʒÿڷ <- subset(newX, split==T)
ڷ <- subset(newX, split==F)
ڷ_group <- ڷ[,8]

lda_train_out <- lda(ü̷~., data=Ʒÿڷ)
str(predict(lda_train_out))
addmargins(xtabs(~Ʒÿڷ$ü̷+predict(lda_train_out)$class), margin=2)
addmargins(prop.table(xtabs(~Ʒÿڷ$ü̷+predict(lda_train_out)$class),margin=1),
           margin=2)

ڷ_pred_group <- predict(lda_train_out, ڷ)
str(ڷ_pred_group)
addmargins(xtabs(~ڷ$ü̷
                 +ڷ_pred_group$class),
           margin=2)
addmargins(prop.table(xtabs(~ڷ$ü̷+ڷ_pred_group$class),margin=1),
           margin=2)

## Ʒÿ  ҵ Ȳ qda() غ...
qda_train_out <- qda(ü̷~., data=Ʒÿڷ)
str(predict(qda_train_out))
addmargins(xtabs(~Ʒÿڷ$ü̷+predict(qda_train_out)$class), margin=2)
addmargins(prop.table(xtabs(~Ʒÿڷ$ü̷+predict(qda_train_out)$class),margin=1),
           margin=2)

ڷ_qda_pred_group <- predict(qda_train_out, ڷ)
str(ڷ_qda_pred_group)
addmargins(xtabs(~ڷ$ü̷
                 +ڷ_qda_pred_group$class),
           margin=2)
addmargins(prop.table(xtabs(~ڷ$ü̷+ڷ_qda_pred_group$class),margin=1),
           margin=2)


## ű ûڿ lda()   ...
predict(lda_out,Y)
table(predict(lda_out,Y)$class)
